module Views.StyleView exposing (styleView)

import Css exposing (..)
import Html.Styled as Html exposing (Html, div, input, label, option, select, text, span)
import Html.Styled.Attributes as Attrs exposing (css, type_, value, selected)
import Html.Styled.Events as Events exposing (onInput, onCheck)
import Model exposing (Model, CenterContent(..), CenterTextConfig, CenterImageConfig)
import State exposing (Msg(..))
import Views.PrimitiveComponents exposing (styledGroup)


styleView : Model -> Html Msg
styleView model =
    div
        [ css
            [ marginTop (px 20)
            , padding (px 8)
            , border3 (px 1) solid (hex "ddd")
            , borderRadius (px 8)
            , backgroundColor (hex "f9f9f9")
            , width (pct 100)
            , maxWidth (pct 100)
            , boxSizing borderBox
            , overflow hidden
            ]
        ]
        [ Html.h3 [] [ text "QR Code Styling" ]
        , colorControls model
        , sizeAndRoundnessControls model
        , centerContentControls model
        ]


colorControls : Model -> Html Msg
colorControls model =
    styledGroup
        []
        [ Html.h4 [] [ text "QR Code Colors" ]
        , div [ css [ displayFlex, flexDirection row ] ]
            [ colorInput "Foreground Color" model.qrStyle.foregroundColor ChangeForegroundColor
            , colorInput "Background Color" model.qrStyle.backgroundColor ChangeBackgroundColor
            ]
        ]


sizeAndRoundnessControls : Model -> Html Msg
sizeAndRoundnessControls model =
    styledGroup
        []
        [ Html.h4 [] [ text "Shape & Transparency" ]
        , div [ css [ displayFlex, flexDirection column ] ]
            [ rangeInput "Dot/Square Size" (String.fromFloat model.qrStyle.moduleScale) "0.5" "1.0" (\s -> ChangeModuleScale (Maybe.withDefault 1.0 (String.toFloat s)))
            , div [ css [ marginTop (px 12) ] ]
                [ rangeInput "Corner Roundness" (String.fromFloat model.qrStyle.cornerRadius) "0" "10" (\s -> ChangeCornerRadius (Maybe.withDefault 0 (String.toFloat s))) ]
            , div [ css [ marginTop (px 12) ] ]
                [ label [] [ text "Make Background Transparent: " ]
                , Html.input
                    [ Attrs.type_ "checkbox"
                    , Attrs.checked (model.qrStyle.backgroundColor == "transparent")
                    , Events.onCheck (\checked -> 
                        if checked then 
                            ChangeBackgroundColor "transparent"
                        else 
                            ChangeBackgroundColor "#FFFFFF"
                    )
                    , css [ marginTop (px 4) ]
                    ]
                    []
                ]
            ]
        ]


centerContentControls : Model -> Html Msg
centerContentControls model =
    styledGroup
        []
        [ Html.h4 [] [ text "Center Content" ]
        , centerContentTypeSelect model
        ,         case model.qrStyle.centerContent of
            NoCenterContent ->
                text ""

            CenterImage config ->
                imageControls config

            CenterText config ->
                textControls model config
        ]


centerContentTypeSelect : Model -> Html Msg
centerContentTypeSelect model =
    div [ css [ marginBottom (px 12) ] ]
        [ label [] [ text "Content Type: " ]
        , select
            [ onInput
                (\value ->
                    case value of
                        "none" ->
                            ChangeCenterContent NoCenterContent

                        "image" ->
                            ChangeCenterContent (CenterImage defaultCenterImageConfig)

                        "text" ->
                            ChangeCenterContent (CenterText defaultCenterTextConfig)

                        _ ->
                            NoOp
                )
            ]
            [ option [ value "none", selected (model.qrStyle.centerContent == NoCenterContent) ] [ text "None" ]
            , option [ value "image", selected (isCenterImage model.qrStyle.centerContent) ] [ text "Image" ]
            , option [ value "text", selected (isCenterText model.qrStyle.centerContent) ] [ text "Text" ]
            ]
        ]


imageControls : CenterImageConfig -> Html Msg
imageControls config =
    div [ css [ marginTop (px 12), displayFlex, flexDirection column ] ]
        [ div [ css [ marginBottom (px 8) ] ]
            [ label [] [ text "Image URL: " ]
            , input
                [ type_ "url"
                , value config.url
                , onInput ChangeCenterImage
                , css [ width (pct 100), padding (px 8), marginTop (px 4) ]
                ]
                []
            ]
        , div [ css [ marginBottom (px 8) ] ]
            [ rangeInput "Image Size (%)" (String.fromInt config.size) "10" "50" (\s -> ChangeCenterImageSize (Maybe.withDefault 25 (String.toInt s))) ]
        , div [ css [ marginBottom (px 8) ] ]
            [ colorInput "Background Color" config.backgroundColor ChangeCenterImageBackgroundColor ]
        , div [ css [ displayFlex, flexDirection row ] ]
            [ colorInput "Border Color" config.borderColor ChangeCenterImageBorderColor
            , div [ css [ flex (int 1) ] ]
                [ label [] [ text "Border Width: " ]
                , input
                    [ type_ "number"
                    , Attrs.step "1"
                    , value (String.fromInt config.borderWidth)
                    , onInput (\s -> ChangeCenterImageBorderWidth (Maybe.withDefault 2 (String.toInt s)))
                    , css [ 
                        property "width" "50px !important"
                      , property "max-width" "50px !important" 
                      , property "min-width" "50px !important"
                      , property "box-sizing" "border-box !important"
                      , property "padding" "4px !important"
                      , property "margin" "0 !important"
                      ]
                    ]
                    []
                ]
            ]
        , div [ css [ marginTop (px 8) ] ]
            [ label [] [ text "Border Radius: " ]
            , input
                [ type_ "number"
                , Attrs.step "1"
                , value (String.fromFloat config.borderRadius)
                , onInput (\s -> ChangeCenterImageBorderRadius (Maybe.withDefault 8.0 (String.toFloat s)))
                , css [ 
                    property "width" "50px !important"
                  , property "max-width" "50px !important" 
                  , property "min-width" "50px !important"
                  , property "box-sizing" "border-box !important"
                  , property "padding" "4px !important"
                  , property "margin" "0 !important"
                  ]
                ]
                []
            ]
        ]


textControls : Model -> CenterTextConfig -> Html Msg
textControls model config =
    div [ css [ marginTop (px 12), displayFlex, flexDirection column, width (pct 100), maxWidth (pct 100), overflow hidden ] ]
        [ div [ css [ backgroundColor (hex "#ff00ff"), border3 (px 5) solid (hex "#00ff00"), padding (px 10), fontSize (px 20) ] ] [ text "🚨 CSS LOADING TEST - IF YOU SEE BRIGHT MAGENTA, CSS WORKS! 🚨" ]
        , div [ css [ width (pct 100), maxWidth (pct 100), marginBottom (px 4), boxSizing borderBox, overflow hidden ] ]
            [ label [] [ text "Text: " ]
            , input
                [ type_ "text"
                , value config.text
                , onInput ChangeCenterText
                , css [ width (pct 100), padding (px 4), marginTop (px 4), boxSizing borderBox, maxWidth (pct 100) ]
                ]
                []
            ]
        , div [ css [ displayFlex, flexDirection row, width (pct 100), maxWidth (pct 100), boxSizing borderBox ] ]
            [ div [ css [ flex (int 1), marginRight (px 2), minWidth (px 0), maxWidth (px 80), overflow hidden ] ]
                [ label [] [ text "Font Size: " ]
                , input
                    [ type_ "number"
                    , Attrs.step "1"
                    , value (String.fromInt config.fontSize)
                    , onInput (\s -> ChangeCenterTextFontSize (Maybe.withDefault 14 (String.toInt s)))
                    , css [ width (pct 100), padding (px 2), marginTop (px 4), boxSizing borderBox, maxWidth (pct 100), minWidth (px 60) ]
                    ]
                    []
                ]
            , div [ css [ flex (int 2), position relative, marginLeft (px 2), minWidth (px 0), maxWidth none, overflow hidden ] ]
                [ label [] [ text "Font Family: " ]
                , div [ css [ width (pct 100), maxWidth (pct 100), overflow hidden ] ]
                    [ div [ css [ backgroundColor (hex "#ffff00"), padding (px 2), marginBottom (px 2), fontSize (px 10) ] ]
                        [ text "🔥 TESTING 🔥" ]
                    , customFontDropdown model config
                    , div [ css [ marginTop (px 4), fontSize (px 10), color (hex "#ff0000"), fontWeight bold ] ]
                        [ text "⚠️ DEBUG: No '🚨 CUSTOM' = cache issue!" ]
                    ]
                ]
            ]
        , div [ css [ displayFlex, flexDirection row, width (pct 100), maxWidth (pct 100), marginTop (px 8), boxSizing borderBox ] ]
            [ div [ css [ flex (int 1), marginRight (px 2), minWidth (px 0), maxWidth (pct 48), overflow hidden ] ]
                [ label [] [ text "Font Weight: " ]
                , select
                    [ onInput ChangeCenterTextFontWeight
                    , css [ width (pct 100), padding (px 2), marginTop (px 4), boxSizing borderBox, maxWidth (pct 100) ]
                    ]
                    [ option [ value "normal", selected (config.fontWeight == "normal") ] [ text "Normal" ]
                    , option [ value "bold", selected (config.fontWeight == "bold") ] [ text "Bold" ]
                    ]
                ]
            , div [ css [ flex (int 1), marginLeft (px 2), minWidth (px 0), maxWidth (pct 48), overflow hidden ] ]
                [ label [] [ text "Font Style: " ]
                , select
                    [ onInput ChangeCenterTextFontStyle
                    , css [ width (pct 100), padding (px 2), marginTop (px 4), boxSizing borderBox, maxWidth (pct 100) ]
                    ]
                    [ option [ value "normal", selected (config.fontStyle == "normal") ] [ text "Normal" ]
                    , option [ value "italic", selected (config.fontStyle == "italic") ] [ text "Italic" ]
                    ]
                ]
            ]
        , div [ css [ displayFlex, flexDirection row, marginTop (px 8), width (pct 100), maxWidth (pct 100), boxSizing borderBox ] ]
            [ div [ css [ flex (int 1), marginRight (px 2), minWidth (px 0), maxWidth (pct 48), overflow hidden ] ]
                [ colorInput "Text Color" config.textColor ChangeCenterTextColor ]
            , div [ css [ flex (int 1), marginLeft (px 2), minWidth (px 0), maxWidth (pct 48), overflow hidden ] ]
                [ colorInput "Background Color" config.backgroundColor ChangeCenterTextBackgroundColor ]
            ]
        , div []
            [ colorInput "Border Color" config.borderColor ChangeCenterTextBorderColor
            , div []
                [ label [] [ text "Border Width: " ]
                , input
                    [ type_ "number"
                    , Attrs.step "1"
                    , value (String.fromInt config.borderWidth)  
                    , onInput (\s -> ChangeCenterTextBorderWidth (Maybe.withDefault 5 (String.toInt s)))
                    , css [ 
                        property "width" "50px !important"
                      , property "max-width" "50px !important" 
                      , property "min-width" "50px !important"
                      , property "box-sizing" "border-box !important"
                      , property "padding" "4px !important"
                      , property "margin" "0 !important"
                      ]
                    ]
                    []
                ]
            , div []
                [ label [] [ text "Padding: " ]
                , input
                    [ type_ "number"
                    , Attrs.step "1"
                    , value (String.fromInt config.padding)
                    , onInput (\s -> ChangeCenterTextPadding (Maybe.withDefault 8 (String.toInt s)))
                    , css [ 
                        property "width" "50px !important"
                      , property "max-width" "50px !important" 
                      , property "min-width" "50px !important"
                      , property "box-sizing" "border-box !important"
                      , property "padding" "4px !important"
                      , property "margin" "0 !important"
                      ]
                    ]
                    []
                ]
            , div []
                [ label [] [ text "Border Radius: " ]
                , input
                    [ type_ "number"
                    , Attrs.step "1"
                    , value (String.fromFloat config.borderRadius)
                    , onInput (\s -> ChangeCenterTextBorderRadius (Maybe.withDefault 4.0 (String.toFloat s)))
                    , css [ 
                        property "width" "50px !important"
                      , property "max-width" "50px !important" 
                      , property "min-width" "50px !important"
                      , property "box-sizing" "border-box !important"
                      , property "padding" "4px !important"
                      , property "margin" "0 !important"
                      ]
                    ]
                    []
                ]
            ]
        ]


-- Custom font dropdown with hover preview
customFontDropdown : Model -> CenterTextConfig -> Html Msg
customFontDropdown model config =
    div [ css [ position relative, width (pct 100), maxWidth (pct 100), minWidth (px 0) ] ]
        [ -- Dropdown trigger button
          div
            [ Events.onClick ToggleFontDropdown
            , css
                [ width (pct 100)
                , padding (px 4)
                , marginTop (px 4)
                , border3 (px 1) solid (hex "#ff0000") -- Red border to show it's working
                , borderRadius (px 4)
                , backgroundColor (hex "#fff")
                , cursor pointer
                , displayFlex
                , justifyContent spaceBetween
                , alignItems center
                , boxSizing borderBox
                , maxWidth (pct 100)
                , overflow hidden
                , textOverflow ellipsis
                , whiteSpace noWrap
                ]
            ]
            [ text ("🚨 CUSTOM: " ++ extractFontName config.fontFamily) -- Shorter prefix
            , text " 🔽"
            ]
        , -- Dropdown options
          if model.fontDropdownOpen then
            div
                [ css
                    [ position absolute
                    , top (pct 100)
                    , left (px 0)
                    , width (pct 100)
                    , maxWidth (pct 100)
                    , maxHeight (px 300)
                    , backgroundColor (hex "#fff")
                    , border3 (px 1) solid (hex "#ccc")
                    , borderRadius (px 4)
                    , boxShadow4 (px 0) (px 4) (px 8) (rgba 0 0 0 0.1)
                    , zIndex (int 1000)
                    , overflowY auto
                    , boxSizing borderBox
                    ]
                ]
                (getSortedFonts config.fontFamily)
          else
            text ""
        , -- Font preview popup
          case model.hoveredFont of
            Nothing ->
                text ""
            
            Just fontFamily ->
                let
                    previewText = if String.isEmpty config.text then "Sample Text" else config.text  
                in
                div
                    [ css
                        [ position fixed -- Use fixed positioning instead of absolute
                        , top (px 100) -- Fixed position from top
                        , right (px 20) -- Fixed position from right
                        , backgroundColor (hex "#ffffff")
                        , border3 (px 3) solid (hex "#ff0000") -- Even brighter red border
                        , borderRadius (px 8)
                        , padding (px 20) -- More padding
                        , boxShadow4 (px 0) (px 12) (px 24) (rgba 0 0 0 0.5) -- Much stronger shadow
                        , zIndex (int 99999) -- Maximum z-index
                        , width (px 250) -- Fixed width
                        , property "font-family" fontFamily
                        , fontSize (px (toFloat config.fontSize))
                        , fontWeight (int (if config.fontWeight == "bold" then 700 else 400))
                        , property "font-style" (if config.fontStyle == "italic" then "italic" else "normal")
                        , color (hex "#000000") -- Always black text for visibility
                        , textAlign center
                        , whiteSpace normal -- Allow text wrapping
                        , overflow visible -- Don't hide content
                        , property "pointer-events" "none" -- Don't interfere with hover events
                        , property "transform" "scale(1.1)" -- Make it slightly larger
                        ]
                    ]
                    [ div [ css [ fontSize (px 14), color (hex "#ff0000"), marginBottom (px 12), fontWeight (int 700) ] ] 
                        [ text ("🎯 FONT PREVIEW: " ++ extractFontName fontFamily) ] -- More obvious debug prefix
                    , div [ css [ fontSize (px (toFloat config.fontSize)), border3 (px 1) solid (hex "#cccccc"), padding (px 8) ] ] 
                        [ text previewText ]
                    ]
        ]


-- Get sorted fonts as dropdown options
getSortedFonts : String -> List (Html Msg)
getSortedFonts selectedFont =
    let
        -- Create a simple test to verify sorting
        testFonts = 
            [ ("Zebra, sans-serif", "Zebra")
            , ("Apple, sans-serif", "Apple")  
            , ("Banana, sans-serif", "Banana")
            ]
        
        fonts =
            -- Sans-serif fonts
            [ ("Arial, sans-serif", "Arial")
            , ("Helvetica, sans-serif", "Helvetica")
            , ("Verdana, sans-serif", "Verdana")
            , ("Tahoma, sans-serif", "Tahoma")
            , ("Trebuchet MS, sans-serif", "Trebuchet MS")
            , ("Geneva, sans-serif", "Geneva")
            , ("Segoe UI, sans-serif", "Segoe UI")
            , ("Calibri, sans-serif", "Calibri")
            , ("Franklin Gothic Medium, sans-serif", "Franklin Gothic")
            , ("Century Gothic, sans-serif", "Century Gothic")
            , ("Futura, sans-serif", "Futura")
            , ("Lato, sans-serif", "Lato")
            , ("Open Sans, sans-serif", "Open Sans")
            , ("Roboto, sans-serif", "Roboto")
            , ("Source Sans Pro, sans-serif", "Source Sans Pro")
            , ("Montserrat, sans-serif", "Montserrat")
            , ("Nunito, sans-serif", "Nunito")
            , ("Ubuntu, sans-serif", "Ubuntu")
            , ("Poppins, sans-serif", "Poppins")
            , ("Inter, sans-serif", "Inter")
            
            -- Serif fonts  
            , ("Times New Roman, serif", "Times New Roman")
            , ("Georgia, serif", "Georgia")
            , ("Palatino, serif", "Palatino")
            , ("Garamond, serif", "Garamond")
            , ("Book Antiqua, serif", "Book Antiqua")
            , ("Times, serif", "Times")
            , ("Baskerville, serif", "Baskerville")
            , ("Cambria, serif", "Cambria")
            , ("Caslon, serif", "Caslon")
            , ("Minion Pro, serif", "Minion Pro")
            , ("Playfair Display, serif", "Playfair Display")
            , ("Merriweather, serif", "Merriweather")
            , ("Crimson Text, serif", "Crimson Text")
            , ("Libre Baskerville, serif", "Libre Baskerville")
            , ("Cormorant Garamond, serif", "Cormorant Garamond")
            
            -- Monospace fonts
            , ("Courier New, monospace", "Courier New")
            , ("Monaco, monospace", "Monaco")
            , ("Consolas, monospace", "Consolas")
            , ("Menlo, monospace", "Menlo")
            , ("Source Code Pro, monospace", "Source Code Pro")
            , ("Fira Code, monospace", "Fira Code")
            , ("JetBrains Mono, monospace", "JetBrains Mono")

            -- Display/Decorative fonts
            , ("Impact, sans-serif", "Impact")
            , ("Arial Black, sans-serif", "Arial Black")
            , ("Oswald, sans-serif", "Oswald")
            , ("Bebas Neue, sans-serif", "Bebas Neue")
            , ("Anton, sans-serif", "Anton")
            , ("Raleway, sans-serif", "Raleway")
            , ("Quicksand, sans-serif", "Quicksand")

            -- Fun/Creative fonts (use sparingly)
            , ("Comic Sans MS, cursive", "Comic Sans MS")
            , ("Brush Script MT, cursive", "Brush Script")
            , ("Pacifico, cursive", "Pacifico")
            ]
            
        sortedFonts = 
            (testFonts ++ fonts)
                |> List.sortBy (\(_, fontDisplay) -> fontDisplay)  -- Sort alphabetically by display name
                
        debugInfo = 
            [ div [ css [ backgroundColor (hex "#00ff00"), color (hex "#000000"), padding (px 4), fontSize (px 10) ] ]
                [ text ("Total fonts: " ++ String.fromInt (List.length sortedFonts)) ]
            , div [ css [ backgroundColor (hex "#ffff00"), color (hex "#000000"), padding (px 4), fontSize (px 10) ] ]
                [ text ("First font: " ++ (sortedFonts |> List.head |> Maybe.map (\(_, name) -> name) |> Maybe.withDefault "None")) ]
            ]
            
        fontElements = 
            sortedFonts
                |> List.map (\(fontValue, fontDisplay) ->
                    div 
                        [ Events.onClick (SelectFont fontValue)
                        , Events.onMouseEnter (FontHover fontValue)
                        , Events.onMouseLeave FontLeave
                        , css
                            [ padding2 (px 8) (px 12)
                            , cursor pointer
                            , property "font-family" fontValue
                            , hover [ backgroundColor (hex "#ffff00") ] -- Bright yellow on hover for debugging
                            , if selectedFont == fontValue then
                                backgroundColor (hex "#e3f2fd")
                              else
                                backgroundColor transparent
                            , border3 (px 1) solid transparent -- Add border space
                            , hover [ border3 (px 1) solid (hex "#ff0000") ] -- Red border on hover for debugging
                            ]
                        ] 
                        [ text (fontDisplay ++ " ★HOVER★") ] -- Very obvious debug text
                )
    in
    debugInfo ++ fontElements


-- Extract font name from font family string (e.g., "Arial, sans-serif" -> "Arial")
extractFontName : String -> String
extractFontName fontFamily =
    fontFamily
        |> String.split ","
        |> List.head
        |> Maybe.withDefault fontFamily
        |> String.trim


-- Helper functions
colorInput : String -> String -> (String -> Msg) -> Html Msg
colorInput labelText currentValue onChangeMsg =
    div [ css [ width (pct 100), minWidth (px 0), maxWidth (pct 100), boxSizing borderBox, overflow hidden ] ]
        [ label [] [ text (labelText ++ ": ") ]
        , if currentValue == "transparent" then
            div
                [ css
                    [ width (pct 100)
                    , height (px 32)
                    , borderRadius (px 4)
                    , marginTop (px 4)
                    , backgroundImage (url "data:image/svg+xml;charset=utf-8,%3csvg width='20' height='20' xmlns='http://www.w3.org/2000/svg'%3e%3cpath d='M0 0h10v10H0zm10 10h10v10H10z' fill='%23f0f0f0'/%3e%3c/svg%3e")
                    , backgroundSize (px 20)
                    , border3 (px 2) solid (hex "ccc")
                    , displayFlex
                    , alignItems center
                    , justifyContent center
                    , fontSize (px 10)
                    , color (hex "666")
                    , cursor pointer
                    , boxSizing borderBox
                    , maxWidth (pct 100)
                    ]
                , Events.onClick (onChangeMsg "#FFFFFF")
                ]
                [ text "Transparent" ]
          else
            input
                [ type_ "color"
                , value currentValue
                , onInput onChangeMsg
                , css [ width (pct 100), height (px 32), border (px 0), borderRadius (px 4), marginTop (px 4), boxSizing borderBox, maxWidth (pct 100) ]
                ]
                []
        ]


rangeInput : String -> String -> String -> String -> (String -> Msg) -> Html Msg
rangeInput labelText currentValue minVal maxVal onChangeMsg =
    div []
        [ label [] [ text (labelText ++ ": " ++ currentValue) ]
        , input
            [ type_ "range"
            , Attrs.min minVal
            , Attrs.max maxVal
            , value currentValue
            , onInput onChangeMsg
            , css [ width (pct 100), marginTop (px 4) ]
            ]
            []
        ]


isCenterImage : CenterContent -> Bool
isCenterImage content =
    case content of
        CenterImage _ ->
            True

        _ ->
            False


isCenterText : CenterContent -> Bool
isCenterText content =
    case content of
        CenterText _ ->
            True

        _ ->
            False


defaultCenterImageConfig : CenterImageConfig
defaultCenterImageConfig =
    { url = ""
    , size = 25
    , backgroundColor = "#FFFFFF"
    , borderColor = "#CCCCCC"
    , borderWidth = 5
    , borderRadius = 8.0
    }


defaultCenterTextConfig : CenterTextConfig
defaultCenterTextConfig =
    { text = ""
    , fontSize = 14
    , fontFamily = "Arial, sans-serif"
    , fontWeight = "normal"
    , fontStyle = "normal"
    , textColor = "#000000"
    , backgroundColor = "#FFFFFF"
    , padding = 8
    , borderRadius = 4.0
    , borderColor = "#CCCCCC"
    , borderWidth = 5
    }
