port module State exposing (..)

import Bytes exposing (Bytes)
import Bytes.Encode
import File.Download as Download
import Image
import Json.Encode as Encode
import Model exposing (Model, QRStyle, CenterContent(..), CenterTextConfig, CenterImageConfig)
import QRCode exposing (defaultImageOptions)
import QRTypes exposing (QRType, encodeQRType)


-- PORTS
port downloadQRCodePNG : String -> Cmd msg


-- Encode QR code data for JavaScript port
encodeQRDataForDownload : Model -> String
encodeQRDataForDownload model =
    let
        qrData = 
            Encode.object
                [ ("qrContent", Encode.string (encodeQRType model.qrType))
                , ("errorCorrection", Encode.string (errorCorrectionToString model.errorCorrection))
                , ("size", Encode.int model.qrStyle.size)
                , ("foregroundColor", Encode.string model.qrStyle.foregroundColor)
                , ("backgroundColor", Encode.string model.qrStyle.backgroundColor)
                , ("cornerRadius", Encode.float model.qrStyle.cornerRadius)
                , ("moduleScale", Encode.float model.qrStyle.moduleScale)
                , ("centerContent", encodeCenterContent model.qrStyle.centerContent)
                ]
    in
    Encode.encode 0 qrData


-- Helper function to convert error correction to string
errorCorrectionToString : QRCode.ErrorCorrection -> String
errorCorrectionToString errorCorrection =
    case errorCorrection of
        QRCode.Low -> "Low"
        QRCode.Medium -> "Medium"
        QRCode.Quartile -> "Quartile"
        QRCode.High -> "High"


-- Helper function to encode center content
encodeCenterContent : CenterContent -> Encode.Value
encodeCenterContent centerContent =
    case centerContent of
        NoCenterContent ->
            Encode.object [ ("type", Encode.string "none") ]
            
        CenterImage config ->
            Encode.object 
                [ ("type", Encode.string "image")
                , ("url", Encode.string config.url)
                , ("size", Encode.int config.size)
                , ("backgroundColor", Encode.string config.backgroundColor)
                , ("borderColor", Encode.string config.borderColor)
                , ("borderWidth", Encode.int config.borderWidth)
                , ("borderRadius", Encode.float config.borderRadius)
                ]
                
        CenterText config ->
            Encode.object 
                [ ("type", Encode.string "text")
                , ("text", Encode.string config.text)
                , ("fontSize", Encode.int config.fontSize)
                , ("fontFamily", Encode.string config.fontFamily)
                , ("fontWeight", Encode.string config.fontWeight)
                , ("fontStyle", Encode.string config.fontStyle)
                , ("textColor", Encode.string config.textColor)
                , ("backgroundColor", Encode.string config.backgroundColor)
                , ("padding", Encode.int config.padding)
                , ("borderRadius", Encode.float config.borderRadius)
                , ("borderColor", Encode.string config.borderColor)
                , ("borderWidth", Encode.int config.borderWidth)
                ]


generateQRCodePngUrl : ( Int, Int ) -> Model -> Bytes
generateQRCodePngUrl ( fg, bg ) model =
    case QRCode.fromStringWith model.errorCorrection (encodeQRType model.qrType) of
        Err _ ->
            Bytes.Encode.encode <| Bytes.Encode.string "ERROR"

        Ok code ->
            Image.toPng
                (QRCode.toImageWithOptions
                    { defaultImageOptions
                        | moduleSize = max 1 (model.qrStyle.size // 25)  -- Responsive module size
                        , darkColor = fg
                        , lightColor = bg
                    }
                    code
                )


type Msg
    = ChangeQRType QRType
    | ChangeErrorCorrection QRCode.ErrorCorrection
    | DownloadQRCodeAsPNG Int Int -- foreground and background color (legacy)
    | DownloadQRCodeWithOverlays -- New download method with overlays
    | ChangeForegroundColor String
    | ChangeBackgroundColor String
    | ChangeCornerRadius Float
    | ChangeModuleScale Float
    | ChangeCenterContent CenterContent
    | ChangeCenterText String
    | ChangeCenterTextFontSize Int
    | ChangeCenterTextFontFamily String
    | ChangeCenterTextColor String
    | ChangeCenterTextBackgroundColor String
    | ChangeCenterTextPadding Int
    | ChangeCenterTextBorderRadius Float
    | ChangeCenterTextBorderColor String
    | ChangeCenterTextBorderWidth Int
    | ChangeCenterTextFontWeight String
    | ChangeCenterTextFontStyle String
    | ChangeCenterImage String
    | ChangeCenterImageSize Int
    | ChangeCenterImageBackgroundColor String
    | ChangeCenterImageBorderColor String
    | ChangeCenterImageBorderWidth Int
    | ChangeCenterImageBorderRadius Float
    | FontHover String
    | FontLeave
    | ToggleFontDropdown
    | SelectFont String
    | NoOp


update : Msg -> Model -> ( Model, Cmd Msg )
update msg model =
    case msg of
        ChangeQRType newType ->
            ( { model | qrType = newType }, Cmd.none )

        ChangeErrorCorrection ecc ->
            ( { model | errorCorrection = ecc }, Cmd.none )

        DownloadQRCodeAsPNG fg bg ->
            ( model, Download.bytes "qrcode.png" "image/png" (generateQRCodePngUrl ( fg, bg ) model) )
            
        DownloadQRCodeWithOverlays ->
            ( model, downloadQRCodePNG (encodeQRDataForDownload model) )

        ChangeForegroundColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | foregroundColor = color }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeBackgroundColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | backgroundColor = color }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCornerRadius radius ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | cornerRadius = radius }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeModuleScale scale ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | moduleScale = scale }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterContent content ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = content }
                
                -- Auto-set error correction to High (30%) when adding center content
                newErrorCorrection = 
                    case (model.qrStyle.centerContent, content) of
                        -- Changing from no content to image/text -> set to High
                        (NoCenterContent, CenterImage _) -> QRCode.High
                        (NoCenterContent, CenterText _) -> QRCode.High
                        -- Keep current error correction for other changes
                        _ -> model.errorCorrection
            in
            ( { model | qrStyle = newStyle, errorCorrection = newErrorCorrection }, Cmd.none )

        ChangeCenterText text ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterText text model.qrStyle.centerContent }
                
                -- Auto-set error correction to High when adding text content
                wasEmpty = case model.qrStyle.centerContent of
                    CenterText config -> String.isEmpty config.text
                    _ -> True
                    
                isNowFilled = not (String.isEmpty text)
                
                newErrorCorrection = 
                    if wasEmpty && isNowFilled then
                        QRCode.High
                    else
                        model.errorCorrection
            in
            ( { model | qrStyle = newStyle, errorCorrection = newErrorCorrection }, Cmd.none )

        ChangeCenterTextFontSize fontSize ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextFontSize fontSize model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextFontFamily fontFamily ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextFontFamily fontFamily model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextColor color model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextBackgroundColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextBackgroundColor color model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextPadding padding ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextPadding padding model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextBorderRadius radius ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextBorderRadius radius model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextBorderColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextBorderColor color model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextBorderWidth width ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextBorderWidth width model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextFontWeight weight ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextFontWeight weight model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterTextFontStyle style ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextFontStyle style model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterImage url ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageUrl url model.qrStyle.centerContent }
                
                -- Auto-set error correction to High when adding image URL
                wasEmpty = case model.qrStyle.centerContent of
                    CenterImage config -> String.isEmpty config.url
                    _ -> True
                    
                isNowFilled = not (String.isEmpty url)
                
                newErrorCorrection = 
                    if wasEmpty && isNowFilled then
                        QRCode.High
                    else
                        model.errorCorrection
            in
            ( { model | qrStyle = newStyle, errorCorrection = newErrorCorrection }, Cmd.none )

        ChangeCenterImageSize size ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageSize size model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterImageBackgroundColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageBackgroundColor color model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterImageBorderColor color ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageBorderColor color model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterImageBorderWidth width ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageBorderWidth width model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        ChangeCenterImageBorderRadius radius ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterImageBorderRadius radius model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle }, Cmd.none )

        FontHover fontFamily ->
            ( { model | hoveredFont = Just fontFamily }, Cmd.none )

        FontLeave ->
            ( { model | hoveredFont = Nothing }, Cmd.none )

        ToggleFontDropdown ->
            ( { model | fontDropdownOpen = not model.fontDropdownOpen }, Cmd.none )

        SelectFont fontFamily ->
            let
                oldStyle = model.qrStyle
                newStyle = { oldStyle | centerContent = updateCenterTextFontFamily fontFamily model.qrStyle.centerContent }
            in
            ( { model | qrStyle = newStyle, fontDropdownOpen = False }, Cmd.none )

        NoOp ->
            ( model, Cmd.none )


-- Helper functions for updating center text configuration
updateCenterText : String -> CenterContent -> CenterContent
updateCenterText text centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | text = text }

        _ ->
            CenterText defaultCenterTextConfig


updateCenterTextFontSize : Int -> CenterContent -> CenterContent
updateCenterTextFontSize fontSize centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | fontSize = fontSize }

        _ ->
            CenterText { defaultCenterTextConfig | fontSize = fontSize }


updateCenterTextFontFamily : String -> CenterContent -> CenterContent
updateCenterTextFontFamily fontFamily centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | fontFamily = fontFamily }

        _ ->
            CenterText { defaultCenterTextConfig | fontFamily = fontFamily }


updateCenterTextColor : String -> CenterContent -> CenterContent
updateCenterTextColor color centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | textColor = color }

        _ ->
            CenterText { defaultCenterTextConfig | textColor = color }


updateCenterTextBackgroundColor : String -> CenterContent -> CenterContent
updateCenterTextBackgroundColor color centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | backgroundColor = color }

        _ ->
            CenterText { defaultCenterTextConfig | backgroundColor = color }


updateCenterTextPadding : Int -> CenterContent -> CenterContent
updateCenterTextPadding padding centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | padding = padding }

        _ ->
            CenterText { defaultCenterTextConfig | padding = padding }


updateCenterTextBorderRadius : Float -> CenterContent -> CenterContent
updateCenterTextBorderRadius radius centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | borderRadius = radius }

        _ ->
            CenterText { defaultCenterTextConfig | borderRadius = radius }


updateCenterTextBorderColor : String -> CenterContent -> CenterContent
updateCenterTextBorderColor color centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | borderColor = color }

        _ ->
            CenterText { defaultCenterTextConfig | borderColor = color }


updateCenterTextBorderWidth : Int -> CenterContent -> CenterContent
updateCenterTextBorderWidth width centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | borderWidth = width }

        _ ->
            CenterText { defaultCenterTextConfig | borderWidth = width }


updateCenterTextFontWeight : String -> CenterContent -> CenterContent
updateCenterTextFontWeight weight centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | fontWeight = weight }

        _ ->
            CenterText { defaultCenterTextConfig | fontWeight = weight }


updateCenterTextFontStyle : String -> CenterContent -> CenterContent
updateCenterTextFontStyle style centerContent =
    case centerContent of
        CenterText config ->
            CenterText { config | fontStyle = style }

        _ ->
            CenterText { defaultCenterTextConfig | fontStyle = style }


updateCenterImageUrl : String -> CenterContent -> CenterContent
updateCenterImageUrl url centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | url = url }

        _ ->
            CenterImage { defaultCenterImageConfig | url = url }


updateCenterImageSize : Int -> CenterContent -> CenterContent
updateCenterImageSize size centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | size = size }

        _ ->
            CenterImage { defaultCenterImageConfig | size = size }


updateCenterImageBackgroundColor : String -> CenterContent -> CenterContent
updateCenterImageBackgroundColor color centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | backgroundColor = color }

        _ ->
            CenterImage { defaultCenterImageConfig | backgroundColor = color }


updateCenterImageBorderColor : String -> CenterContent -> CenterContent
updateCenterImageBorderColor color centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | borderColor = color }

        _ ->
            CenterImage { defaultCenterImageConfig | borderColor = color }


updateCenterImageBorderWidth : Int -> CenterContent -> CenterContent
updateCenterImageBorderWidth width centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | borderWidth = width }

        _ ->
            CenterImage { defaultCenterImageConfig | borderWidth = width }


updateCenterImageBorderRadius : Float -> CenterContent -> CenterContent
updateCenterImageBorderRadius radius centerContent =
    case centerContent of
        CenterImage config ->
            CenterImage { config | borderRadius = radius }

        _ ->
            CenterImage { defaultCenterImageConfig | borderRadius = radius }


defaultCenterImageConfig : CenterImageConfig
defaultCenterImageConfig =
    { url = ""
    , size = 25  -- 25% of QR code size
    , backgroundColor = "#FFFFFF"
    , borderColor = "#CCCCCC"
    , borderWidth = 5
    , borderRadius = 8.0
    }


defaultCenterTextConfig : CenterTextConfig
defaultCenterTextConfig =
    { text = ""
    , fontSize = 14
    , fontFamily = "Arial, sans-serif"
    , fontWeight = "normal"
    , fontStyle = "normal"
    , textColor = "#000000"
    , backgroundColor = "#FFFFFF"
    , padding = 8
    , borderRadius = 4.0
    , borderColor = "#CCCCCC"
    , borderWidth = 5
    }
