// QR Code Download with Overlays
(function() {
  'use strict';

  // Initialize the port connection when Elm app starts
  function initializeQRDownload(app) {
    if (!app.ports || !app.ports.downloadQRCodePNG) {
      console.warn('downloadQRCodePNG port not found');
      return;
    }

    app.ports.downloadQRCodePNG.subscribe(function(qrDataJson) {
      try {
        const qrData = JSON.parse(qrDataJson);
        console.log('QR Data received:', qrData);
        
        // Add a small delay to ensure DOM is fully rendered
        setTimeout(() => {
          generateAndDownloadQR(qrData);
        }, 100);
        
      } catch (error) {
        console.error('Error parsing QR data:', error);
        console.error('Raw data received:', qrDataJson);
      }
    });
  }

  // Generate QR code and download as PNG
  async function generateAndDownloadQR(qrData) {
    try {
      console.log('Starting QR code capture...');
      
      // Find all possible SVG elements and their containers
      const allSvgs = document.querySelectorAll('svg');
      console.log('Found', allSvgs.length, 'SVG elements');

      // Find the actual QR code container in the DOM
      // Try multiple selectors to find the QR container
      let qrContainer = null;
      let svgElement = null;

      // Method 1: Try to find container with padding (30px now)
      const containers = document.querySelectorAll('div');
      for (const container of containers) {
        const style = getComputedStyle(container);
        const svg = container.querySelector('svg');
        if (svg && (style.padding.includes('30px') || style.padding.includes('20px'))) {
          qrContainer = container;
          svgElement = svg;
          console.log('Found QR container with padding:', style.padding);
          break;
        }
      }

      // Method 2: Find container by looking for QR-specific styling
      if (!qrContainer) {
        for (const container of containers) {
          const style = getComputedStyle(container);
          const svg = container.querySelector('svg');
          if (svg && (style.backgroundColor === 'rgb(255, 255, 255)' || style.backgroundColor === 'white')) {
            const rect = container.getBoundingClientRect();
            if (rect.width > 100 && rect.height > 100) {
              qrContainer = container;
              svgElement = svg;
              console.log('Found QR container by background and size');
              break;
            }
          }
        }
      }

      // Method 3: Fallback - find largest SVG
      if (!svgElement) {
        let largestSvg = null;
        let largestSize = 0;
        
        for (const svg of allSvgs) {
          const rect = svg.getBoundingClientRect();
          const size = rect.width * rect.height;
          if (size > largestSize && rect.width > 50 && rect.height > 50) {
            largestSize = size;
            largestSvg = svg;
          }
        }
        
        if (largestSvg) {
          svgElement = largestSvg;
          qrContainer = largestSvg.parentElement;
          console.log('Found QR SVG by largest size:', largestSize);
        }
      }
      
      if (!svgElement) {
        console.error('Could not find QR code SVG element in DOM');
        alert('Could not find QR code to download. Please try again.');
        return;
      }
      
      console.log('Using container:', qrContainer);
      console.log('Using SVG:', svgElement);

      // Get the actual rendered size including padding
      const containerRect = qrContainer.getBoundingClientRect();
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      
      // HIGH RESOLUTION: Scale up the canvas for much better quality
      // This allows the QR code to be blown up without looking grainy
      const highResScale = 5; // 5x resolution for ultra-sharp printing
      const bufferSize = 20; // Increase buffer to ensure no clipping
      canvas.width = (Math.ceil(containerRect.width) + (bufferSize * 2)) * highResScale;
      canvas.height = (Math.ceil(containerRect.height) + (bufferSize * 2)) * highResScale;

      console.log('Canvas size:', canvas.width, 'x', canvas.height);
      console.log('Container size:', containerRect.width, 'x', containerRect.height);
      console.log('Buffer size:', bufferSize);
      console.log('High-res scale:', highResScale);

      // Capture the actual rendered QR code with buffer offset and high-res scale
      await captureQRCodeToCanvas(canvas, ctx, qrContainer, bufferSize, highResScale);

      // Convert canvas to blob and download
      canvas.toBlob(function(blob) {
        if (!blob) {
          console.error('Failed to create blob from canvas');
          alert('Failed to generate PNG. Canvas may be too large or empty.');
          return;
        }
        
        console.log('Blob created successfully, size:', blob.size, 'bytes');
        
        try {
          const url = URL.createObjectURL(blob);
          const a = document.createElement('a');
          a.href = url;
          a.download = 'qrcode.png';
          document.body.appendChild(a);
          a.click();
          document.body.removeChild(a);
          URL.revokeObjectURL(url);
          console.log('Download initiated successfully');
          
        } catch (downloadError) {
          console.error('Error during download:', downloadError);
          alert('Error during download: ' + downloadError.message);
        }
      }, 'image/png');

    } catch (error) {
      console.error('Error generating QR code:', error);
      
      // Provide more specific error messages based on the type of error
      const errorMsg = error.message || 'Unknown error';
      
      if (errorMsg.includes('CORS') || errorMsg.includes('cross-origin') || errorMsg.includes('tainted')) {
        alert('Canvas was tainted by external image. This should not happen with the new code. Please report this bug.');
      } else if (errorMsg.includes('canvas') || errorMsg.includes('Canvas')) {
        alert('Canvas rendering error. Please try again or refresh the page.');
      } else if (errorMsg.includes('Image') || errorMsg.includes('image')) {
        alert('Image processing error. The QR code downloaded with a smart placeholder icon based on your image type.');
      } else if (errorMsg.includes('roundRect')) {
        alert('Browser does not support rounded rectangles. The QR code downloaded successfully with square borders.');
      } else {
        alert('Error generating QR code: ' + errorMsg + '. Please try refreshing the page.');
      }
    }
  }

  // Capture the actual QR code from the DOM to canvas
  async function captureQRCodeToCanvas(canvas, ctx, container, bufferOffset = 0, scale = 1) {
    console.log('Capturing QR code from DOM element with buffer:', bufferOffset, 'scale:', scale);

    try {
      // Scale the canvas context for high resolution
      ctx.scale(scale, scale);
      
      // Get computed styles
      const containerStyles = getComputedStyle(container);
      const containerRect = container.getBoundingClientRect();

      console.log('Container background:', containerStyles.backgroundColor);
      console.log('Container rect:', containerRect);

      // Clear canvas with white background
      ctx.clearRect(0, 0, canvas.width / scale, canvas.height / scale);
      ctx.fillStyle = '#ffffff';
      ctx.fillRect(0, 0, canvas.width / scale, canvas.height / scale);

      // Draw container background in the buffered area
      ctx.fillStyle = containerStyles.backgroundColor || '#ffffff';
      ctx.fillRect(bufferOffset, bufferOffset, containerRect.width, containerRect.height);
      console.log('Container background drawn');
    } catch (error) {
      console.error('Error setting up canvas:', error);
      throw error;
    }
    
    // Find and render the SVG
    try {
      const svgElement = container.querySelector('svg');
      if (svgElement) {
        console.log('Found SVG element, rendering...');
        await renderSVGToCanvas(ctx, svgElement, container, bufferOffset);
        console.log('SVG rendered successfully');
      } else {
        console.warn('No SVG element found in container');
      }
    } catch (error) {
      console.error('Error rendering SVG:', error);
      throw error;
    }
    
    // Find and render any overlay elements (text/images)
    // Try multiple ways to find overlays
    const overlays = [
      ...container.querySelectorAll('div[style*="position: absolute"]'),
      ...container.querySelectorAll('img[style*="position: absolute"]'),
      ...container.querySelectorAll('div[style*="position:absolute"]'), // No space variant
      ...container.querySelectorAll('img[style*="position:absolute"]')   // No space variant
    ];
    
    // Also look for elements with computed absolute positioning
    const allDivs = container.querySelectorAll('div');
    for (const div of allDivs) {
      const computedStyle = getComputedStyle(div);
      if (computedStyle.position === 'absolute' && div.textContent && div.textContent.trim()) {
        overlays.push(div);
      }
    }
    
    // Also look for all images with computed absolute positioning
    const allImages = container.querySelectorAll('img');
    for (const img of allImages) {
      const computedStyle = getComputedStyle(img);
      if (computedStyle.position === 'absolute') {
        overlays.push(img);
      }
    }
    
    // Also look for any absolutely positioned elements (catch-all)
    const allElements = container.querySelectorAll('*');
    for (const element of allElements) {
      const computedStyle = getComputedStyle(element);
      if (computedStyle.position === 'absolute' && !overlays.includes(element)) {
        overlays.push(element);
      }
    }
    
    console.log('Found', overlays.length, 'overlay elements');
    overlays.forEach((overlay, i) => {
      console.log(`Overlay ${i}:`, overlay.textContent || overlay.src, getComputedStyle(overlay).position);
    });
    
    // Render overlays one by one, continuing even if some fail
    let successfulOverlays = 0;
    for (let i = 0; i < overlays.length; i++) {
      try {
        console.log(`Rendering overlay ${i + 1}/${overlays.length}`);
        await renderOverlayToCanvas(ctx, overlays[i], container, bufferOffset);
        console.log(`Successfully rendered overlay ${i + 1}`);
        successfulOverlays++;
      } catch (overlayError) {
        console.error(`Error rendering overlay ${i + 1}:`, overlayError);
        console.log(`Continuing with remaining overlays...`);
        // Continue with other overlays instead of failing completely
      }
    }
    console.log(`Overlay rendering completed: ${successfulOverlays}/${overlays.length} successful`);
    
    // Provide user feedback about image processing
    if (overlays.length > 0) {
      console.log('🎯 Image Processing Summary:');
      let corsSuccessCount = 0;
      let placeholderCount = 0;
      let directDrawCount = 0;
      
      // This is a summary - the actual tracking would need to be done during processing
      // For now, just provide general feedback
      if (successfulOverlays === overlays.length) {
        console.log('✅ All overlays rendered successfully');
      } else {
        console.log(`⚠️ ${overlays.length - successfulOverlays} overlays failed to render`);
      }
    }
  }

  // Render SVG element to canvas
  async function renderSVGToCanvas(ctx, svgElement, container, bufferOffset = 0) {
    try {
      // Get SVG data
      const svgData = new XMLSerializer().serializeToString(svgElement);
      const svgBlob = new Blob([svgData], { type: 'image/svg+xml;charset=utf-8' });
      const url = URL.createObjectURL(svgBlob);

      // Create image from SVG
      const img = new Image();
      await new Promise((resolve, reject) => {
        img.onload = resolve;
        img.onerror = reject;
        img.src = url;
      });

      // Calculate position within container (accounting for padding and buffer)
      const containerStyles = getComputedStyle(container);
      const paddingLeft = parseInt(containerStyles.paddingLeft) || 0;
      const paddingTop = parseInt(containerStyles.paddingTop) || 0;

      // Draw the SVG image with buffer offset
      ctx.drawImage(img, bufferOffset + paddingLeft, bufferOffset + paddingTop);

      // Clean up
      URL.revokeObjectURL(url);

    } catch (error) {
      console.error('Error rendering SVG to canvas:', error);
    }
  }

  // Render overlay elements (text, images) to canvas
  async function renderOverlayToCanvas(ctx, overlay, container, bufferOffset = 0) {
    try {
      const overlayStyles = getComputedStyle(overlay);
      const containerRect = container.getBoundingClientRect();
      const overlayRect = overlay.getBoundingClientRect();

      // Calculate relative position with buffer offset
      const x = bufferOffset + (overlayRect.left - containerRect.left);
      const y = bufferOffset + (overlayRect.top - containerRect.top);
      
      if (overlay.tagName === 'IMG') {
        // Render image overlay
        console.log('Processing image overlay:', overlay.src, 'at position', x, y, 'size:', overlayRect.width, 'x', overlayRect.height);
        console.log('Image type detected:', overlay.src.startsWith('data:') ? 'Data URI' : overlay.src.startsWith('blob:') ? 'Blob URL' : 'Regular URL');
        
        // Check if this is an external image that would taint the canvas
        const isExternalImage = overlay.src.startsWith('blob:') || 
                               overlay.src.startsWith('http://') || 
                               overlay.src.startsWith('https://') ||
                               (!overlay.src.startsWith('data:') && !overlay.src.startsWith('/'));
        
        if (isExternalImage) {
          console.log('🚫 External image detected - trying to load with CORS first, then fallback to smart placeholder');
          
          // First attempt: Try to load the external image with proper CORS handling
          try {
            const proxyImg = new Image();
            proxyImg.crossOrigin = 'anonymous';
            
            await new Promise((resolve, reject) => {
              const timeout = setTimeout(() => reject(new Error('CORS load timeout')), 3000);
              
              proxyImg.onload = () => {
                clearTimeout(timeout);
                console.log('✅ Successfully loaded external image with CORS');
                resolve();
              };
              
              proxyImg.onerror = () => {
                clearTimeout(timeout);
                reject(new Error('CORS load failed'));
              };
              
              proxyImg.src = overlay.src;
            });
            
            // If we get here, the image loaded successfully with CORS
            ctx.drawImage(proxyImg, x, y, overlayRect.width, overlayRect.height);
            console.log('✅ Drew external image successfully with CORS');
            
          } catch (corsError) {
            console.log('⚠️ CORS failed, drawing smart placeholder:', corsError.message);
            
            // CORS failed, draw smart placeholder based on image URL/type
            const computedStyles = getComputedStyle(overlay);
            
            // Draw background matching the original image's background
            ctx.fillStyle = computedStyles.backgroundColor || '#ffffff';
            ctx.fillRect(x, y, overlayRect.width, overlayRect.height);
            
            // Draw border if present  
            const borderWidth = parseInt(computedStyles.borderWidth) || 0;
            if (borderWidth > 0) {
              ctx.strokeStyle = computedStyles.borderColor || '#cccccc';
              ctx.lineWidth = borderWidth;
              ctx.strokeRect(x + borderWidth/2, y + borderWidth/2, overlayRect.width - borderWidth, overlayRect.height - borderWidth);
            }
            
            // Determine appropriate icon based on image URL/type
            const imageUrl = overlay.src.toLowerCase();
            let icon = '🖼️'; // Default image icon
            
            if (imageUrl.includes('wifi') || imageUrl.includes('wireless') || imageUrl.includes('signal')) {
              icon = '📶'; // WiFi signal
            } else if (imageUrl.includes('qr') || imageUrl.includes('code')) {
              icon = '⬜'; // QR code
            } else if (imageUrl.includes('logo') || imageUrl.includes('brand')) {
              icon = '🏷️'; // Logo/brand
            } else if (imageUrl.includes('user') || imageUrl.includes('person') || imageUrl.includes('avatar')) {
              icon = '👤'; // User/person
            } else if (imageUrl.includes('phone') || imageUrl.includes('mobile')) {
              icon = '📱'; // Phone
            } else if (imageUrl.includes('email') || imageUrl.includes('mail')) {
              icon = '📧'; // Email
            } else if (imageUrl.includes('location') || imageUrl.includes('map') || imageUrl.includes('gps')) {
              icon = '📍'; // Location
            } else if (imageUrl.includes('web') || imageUrl.includes('www') || imageUrl.includes('http')) {
              icon = '🌐'; // Web/URL
            } else if (imageUrl.includes('social') || imageUrl.includes('facebook') || imageUrl.includes('twitter') || imageUrl.includes('instagram')) {
              icon = '📲'; // Social media
            } else if (imageUrl.includes('business') || imageUrl.includes('company')) {
              icon = '🏢'; // Business
            }
            
            // Draw the smart icon
            ctx.fillStyle = '#666666';
            const iconSize = Math.min(overlayRect.width, overlayRect.height) * 0.6;
            ctx.font = `${iconSize}px Arial`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.fillText(icon, x + overlayRect.width/2, y + overlayRect.height/2);
            
            console.log(`✅ Drew smart placeholder icon "${icon}" for URL: ${imageUrl}`);
          }
          
        } else {
          // Safe to draw data URIs and same-origin images
          console.log('✅ Safe image detected - drawing directly');
          
          try {
            ctx.drawImage(overlay, x, y, overlayRect.width, overlayRect.height);
            console.log('Successfully drew safe image directly');
            
          } catch (drawError) {
            console.log('Direct draw failed, trying new image load:', drawError.message);
            
            try {
              const img = new Image();
              img.src = overlay.src;
              
              await new Promise((resolve, reject) => {
                const timeout = setTimeout(() => reject(new Error('Timeout')), 2000);
                img.onload = () => { clearTimeout(timeout); resolve(); };
                img.onerror = () => { clearTimeout(timeout); reject(new Error('Load failed')); };
              });
              
              ctx.drawImage(img, x, y, overlayRect.width, overlayRect.height);
              console.log('Successfully drew image after reload');
              
            } catch (reloadError) {
              console.log('Image reload failed, drawing placeholder:', reloadError.message);
              
              // Draw simple placeholder
              ctx.fillStyle = '#f0f0f0';
              ctx.fillRect(x, y, overlayRect.width, overlayRect.height);
              ctx.strokeStyle = '#cccccc';
              ctx.strokeRect(x, y, overlayRect.width, overlayRect.height);
              ctx.fillStyle = '#666666';
              ctx.font = '16px Arial';
              ctx.textAlign = 'center';
              ctx.textBaseline = 'middle';
              ctx.fillText('🖼️', x + overlayRect.width/2, y + overlayRect.height/2);
            }
          }
        }
        
      } else if (overlay.textContent && overlay.textContent.trim()) {
        // Render text overlay
        const text = overlay.textContent.trim();
        const fontSize = parseInt(overlayStyles.fontSize) || 14;
        const fontFamily = overlayStyles.fontFamily || 'Arial';
        const fontWeight = overlayStyles.fontWeight || 'normal';
        const fontStyle = overlayStyles.fontStyle || 'normal';
        const textColor = overlayStyles.color || '#000000';
        const bgColor = overlayStyles.backgroundColor;
        const borderRadius = parseFloat(overlayStyles.borderRadius) || 0;
        const padding = parseInt(overlayStyles.padding) || 8;
        
        console.log('Rendering text overlay:', text, 'at', x, y, overlayRect.width, overlayRect.height);
        console.log('Text styles:', { fontSize, fontFamily, fontWeight, textColor, bgColor });
        
        // Set font
        ctx.font = `${fontStyle} ${fontWeight} ${fontSize}px ${fontFamily}`;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        
        // Draw background with border radius if present
        if (bgColor && bgColor !== 'rgba(0, 0, 0, 0)' && bgColor !== 'transparent') {
          ctx.fillStyle = bgColor;
          if (borderRadius > 0) {
            drawRoundedRect(ctx, x, y, overlayRect.width, overlayRect.height, borderRadius);
          } else {
            ctx.fillRect(x, y, overlayRect.width, overlayRect.height);
          }
        }
        
        // Draw border if present
        const borderWidth = parseInt(overlayStyles.borderWidth) || 0;
        const borderColor = overlayStyles.borderColor;
        if (borderWidth > 0 && borderColor && borderColor !== 'transparent') {
          ctx.strokeStyle = borderColor;
          ctx.lineWidth = borderWidth;
          if (borderRadius > 0) {
            drawRoundedRectStroke(ctx, x, y, overlayRect.width, overlayRect.height, borderRadius);
          } else {
            ctx.strokeRect(x, y, overlayRect.width, overlayRect.height);
          }
        }
        
        // Draw text
        ctx.fillStyle = textColor;
        ctx.fillText(text, x + overlayRect.width / 2, y + overlayRect.height / 2);
      }
      
    } catch (error) {
      console.error('Error rendering overlay to canvas:', error);
    }
  }

  // Draw center content (text or image)
  async function drawCenterContent(ctx, centerContent, qrSize) {
    const centerX = qrSize / 2;
    const centerY = qrSize / 2;
    
    if (centerContent.type === 'text' && centerContent.text) {
      drawCenterText(ctx, centerContent, centerX, centerY);
    } else if (centerContent.type === 'image' && centerContent.url) {
      await drawCenterImage(ctx, centerContent, centerX, centerY, qrSize);
    }
  }

  // Draw center text
  function drawCenterText(ctx, config, centerX, centerY) {
    // Set font
    const fontStyle = config.fontStyle === 'italic' ? 'italic ' : '';
    const fontWeight = config.fontWeight === 'bold' ? 'bold ' : '';
    const fontSize = config.fontSize || 14;
    const fontFamily = config.fontFamily || 'Arial, sans-serif';
    ctx.font = `${fontStyle}${fontWeight}${fontSize}px ${fontFamily}`;
    
    // Measure text
    const metrics = ctx.measureText(config.text);
    const textWidth = metrics.width;
    const textHeight = fontSize;
    
    // Calculate background dimensions
    const padding = config.padding || 8;
    const bgWidth = textWidth + (padding * 2);
    const bgHeight = textHeight + (padding * 2);
    
    // Draw background
    if (config.backgroundColor) {
      ctx.fillStyle = config.backgroundColor;
      const bgX = centerX - bgWidth / 2;
      const bgY = centerY - bgHeight / 2;
      
      if (config.borderRadius > 0) {
        drawRoundedRect(ctx, bgX, bgY, bgWidth, bgHeight, config.borderRadius);
      } else {
        ctx.fillRect(bgX, bgY, bgWidth, bgHeight);
      }
    }
    
    // Draw border
    if (config.borderWidth > 0 && config.borderColor) {
      ctx.strokeStyle = config.borderColor;
      ctx.lineWidth = config.borderWidth;
      const bgX = centerX - bgWidth / 2;
      const bgY = centerY - bgHeight / 2;
      
      if (config.borderRadius > 0) {
        drawRoundedRectStroke(ctx, bgX, bgY, bgWidth, bgHeight, config.borderRadius);
      } else {
        ctx.strokeRect(bgX, bgY, bgWidth, bgHeight);
      }
    }
    
    // Draw text
    ctx.fillStyle = config.textColor || '#000000';
    ctx.textAlign = 'center';
    ctx.textBaseline = 'middle';
    ctx.fillText(config.text, centerX, centerY);
  }

  // Draw center image
  async function drawCenterImage(ctx, config, centerX, centerY, qrSize) {
    try {
      const img = new Image();
      img.crossOrigin = 'anonymous';
      
      await new Promise((resolve, reject) => {
        img.onload = resolve;
        img.onerror = reject;
        img.src = config.url;
      });
      
      // Calculate image dimensions
      const maxSize = (qrSize * config.size) / 100; // size is percentage
      const aspectRatio = img.width / img.height;
      let imgWidth, imgHeight;
      
      if (aspectRatio > 1) {
        imgWidth = maxSize;
        imgHeight = maxSize / aspectRatio;
      } else {
        imgHeight = maxSize;
        imgWidth = maxSize * aspectRatio;
      }
      
      // Add padding for background
      const padding = 6;
      const bgWidth = imgWidth + (padding * 2);
      const bgHeight = imgHeight + (padding * 2);
      
      // Draw background
      if (config.backgroundColor) {
        ctx.fillStyle = config.backgroundColor;
        const bgX = centerX - bgWidth / 2;
        const bgY = centerY - bgHeight / 2;
        
        if (config.borderRadius > 0) {
          drawRoundedRect(ctx, bgX, bgY, bgWidth, bgHeight, config.borderRadius);
        } else {
          ctx.fillRect(bgX, bgY, bgWidth, bgHeight);
        }
      }
      
      // Draw border
      if (config.borderWidth > 0 && config.borderColor) {
        ctx.strokeStyle = config.borderColor;
        ctx.lineWidth = config.borderWidth;
        const bgX = centerX - bgWidth / 2;
        const bgY = centerY - bgHeight / 2;
        
        if (config.borderRadius > 0) {
          drawRoundedRectStroke(ctx, bgX, bgY, bgWidth, bgHeight, config.borderRadius);
        } else {
          ctx.strokeRect(bgX, bgY, bgWidth, bgHeight);
        }
      }
      
      // Draw image
      ctx.drawImage(img, 
        centerX - imgWidth / 2, 
        centerY - imgHeight / 2, 
        imgWidth, 
        imgHeight
      );
      
    } catch (error) {
      console.warn('Failed to load center image:', error);
    }
  }

  // Helper function to draw rounded rectangle
  function drawRoundedRect(ctx, x, y, width, height, radius) {
    ctx.beginPath();
    ctx.moveTo(x + radius, y);
    ctx.lineTo(x + width - radius, y);
    ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
    ctx.lineTo(x + width, y + height - radius);
    ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    ctx.lineTo(x + radius, y + height);
    ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
    ctx.lineTo(x, y + radius);
    ctx.quadraticCurveTo(x, y, x + radius, y);
    ctx.closePath();
    ctx.fill();
  }

  // Helper function to draw rounded rectangle stroke
  function drawRoundedRectStroke(ctx, x, y, width, height, radius) {
    ctx.beginPath();
    ctx.moveTo(x + radius, y);
    ctx.lineTo(x + width - radius, y);
    ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
    ctx.lineTo(x + width, y + height - radius);
    ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    ctx.lineTo(x + radius, y + height);
    ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
    ctx.lineTo(x, y + radius);
    ctx.quadraticCurveTo(x, y, x + radius, y);
    ctx.closePath();
    ctx.stroke();
  }

  // Convert error correction level string to QRCode.js format
  function getErrorCorrectionLevel(level) {
    switch (level) {
      case 'Low': return 'L';
      case 'Medium': return 'M';
      case 'Quartile': return 'Q';
      case 'High': return 'H';
      default: return 'M';
    }
  }

  // Helper function to draw rounded rectangle
  function drawRoundedRect(ctx, x, y, width, height, radius) {
    ctx.beginPath();
    ctx.moveTo(x + radius, y);
    ctx.lineTo(x + width - radius, y);
    ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
    ctx.lineTo(x + width, y + height - radius);
    ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    ctx.lineTo(x + radius, y + height);
    ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
    ctx.lineTo(x, y + radius);
    ctx.quadraticCurveTo(x, y, x + radius, y);
    ctx.closePath();
    ctx.fill();
  }

  // Helper function to stroke rounded rectangle
  function drawRoundedRectStroke(ctx, x, y, width, height, radius) {
    ctx.beginPath();
    ctx.moveTo(x + radius, y);
    ctx.lineTo(x + width - radius, y);
    ctx.quadraticCurveTo(x + width, y, x + width, y + radius);
    ctx.lineTo(x + width, y + height - radius);
    ctx.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    ctx.lineTo(x + radius, y + height);
    ctx.quadraticCurveTo(x, y + height, x, y + height - radius);
    ctx.lineTo(x, y + radius);
    ctx.quadraticCurveTo(x, y, x + radius, y);
    ctx.closePath();
    ctx.stroke();
  }

  // Advanced method to draw images while avoiding CORS issues
  async function drawImageViaCanvas(ctx, imgElement, x, y, width, height) {
    console.log('Attempting canvas-based image drawing for CORS avoidance');
    console.log('Image src:', imgElement.src);
    console.log('Image complete:', imgElement.complete);
    console.log('Image natural dimensions:', imgElement.naturalWidth, 'x', imgElement.naturalHeight);
    
    try {
      // Create a temporary canvas to capture the image
      const tempCanvas = document.createElement('canvas');
      const tempCtx = tempCanvas.getContext('2d');
      
      // Set canvas size to match the image display size
      tempCanvas.width = imgElement.naturalWidth || imgElement.width || width;
      tempCanvas.height = imgElement.naturalHeight || imgElement.height || height;
      
      console.log('Temp canvas size:', tempCanvas.width, 'x', tempCanvas.height);
      
      // Try to draw the image to the temporary canvas
      try {
        tempCtx.drawImage(imgElement, 0, 0);
        console.log('Successfully drew image to temporary canvas');
        
        // Now draw from the temporary canvas to the main canvas
        ctx.drawImage(tempCanvas, x, y, width, height);
        console.log('Successfully transferred image from temp canvas to main canvas');
        
      } catch (tempCanvasError) {
        console.log('Temp canvas method failed:', tempCanvasError.message);
        
        // Try alternative: direct drawing as last resort
        if (imgElement.complete && imgElement.naturalWidth > 0) {
          console.log('Trying direct element drawing as last resort');
          ctx.drawImage(imgElement, x, y, width, height);
          console.log('Direct drawing succeeded');
        } else {
          throw new Error('Image not ready for drawing - not complete or no dimensions');
        }
      }
      
    } catch (error) {
      console.error('Canvas-based image drawing failed:', error);
      throw error;
    }
  }

  // Make the function available globally
  window.initializeQRDownload = initializeQRDownload;

})();
